import React from "react";
import { useLanguage } from "@/contexts/LanguageContext";

type TaskAnalysisData = {
  taskStats: {
    totalTasks: number;
    completedTasks: number;
    pendingTasks: number;
    overdueTasks: number;
    recurringTasks: number;
    nonRecurringTasks: number;
    periodCompletionRate: number;
    totalPeriods: number;
    completedPeriods: number;
    tasksCreatedToday: number;
    tasksCreatedThisWeek: number;
    tasksCreatedThisMonth: number;
  };
  patternStats: Array<{
    pattern: string;
    total: number;
    completed: number;
    completionRate: number;
  }>;
  departmentStats: Array<{
    department: string;
    total: number;
    completed: number;
    overdue: number;
    nonConformities: number;
    completionRate: number;
  }>;
  technicianStats: Array<{
    technicianId: string;
    technicianName: string;
    technicianEmail: string;
    isExternal: boolean;
    companyName: string | null;
    total: number;
    completed: number;
    overdue: number;
    avgCompletionTime: number;
    completionRate: number;
  }>;
  nonConformityStats: {
    totalNonConformities: number;
    tasksWithNonConformities: number;
    nonConformityRate: number;
  };
};

interface TaskCompletionTabProps {
  taskData: TaskAnalysisData;
}

export default function TaskCompletionTab({ taskData }: TaskCompletionTabProps) {
  const { t } = useLanguage();
  const { departmentStats, technicianStats } = taskData;

  // Sort by completion rate descending
  const sortedDepartments = [...departmentStats].sort(
    (a, b) => b.completionRate - a.completionRate
  );

  const sortedTechnicians = [...technicianStats].sort(
    (a, b) => b.completionRate - a.completionRate
  );

  return (
    <>
      {/* Department Completion */}
      <div className='bg-white p-6 rounded-lg shadow mb-8'>
        <h3 className='text-lg font-semibold mb-4'>
          {t("analysis.taskAnalysis.completionByDepartment") || "Completion by Department"}
        </h3>
        <div className='overflow-x-auto'>
          <table className='min-w-full divide-y divide-gray-200'>
            <thead className='bg-gray-50'>
              <tr>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.department") || "Department"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.total") || "Total"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completed") || "Completed"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.overdue") || "Overdue"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completionRateLabel") || "Completion Rate"}
                </th>
              </tr>
            </thead>
            <tbody className='bg-white divide-y divide-gray-200'>
              {sortedDepartments.map((dept) => (
                <tr key={dept.department}>
                  <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900'>
                    {dept.department}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {dept.total}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {dept.completed}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-red-600'>
                    {dept.overdue}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm'>
                    <div className='flex items-center gap-2'>
                      <div className='flex-1 bg-gray-200 rounded-full h-2 max-w-[100px]'>
                        <div
                          className={`h-2 rounded-full ${
                            dept.completionRate >= 80
                              ? "bg-green-500"
                              : dept.completionRate >= 50
                              ? "bg-orange-500"
                              : "bg-red-500"
                          }`}
                          style={{ width: `${Math.min(dept.completionRate, 100)}%` }}
                        />
                      </div>
                      <span
                        className={`font-medium ${
                          dept.completionRate >= 80
                            ? "text-green-600"
                            : dept.completionRate >= 50
                            ? "text-orange-600"
                            : "text-red-600"
                        }`}
                      >
                        {dept.completionRate.toFixed(1)}%
                      </span>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Technician Completion */}
      <div className='bg-white p-6 rounded-lg shadow'>
        <h3 className='text-lg font-semibold mb-4'>
          {t("analysis.taskAnalysis.completionByTechnician") || "Completion by Technician"}
        </h3>
        <div className='overflow-x-auto'>
          <table className='min-w-full divide-y divide-gray-200'>
            <thead className='bg-gray-50'>
              <tr>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.technician") || "Technician"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.type") || "Type"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.total") || "Total"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completed") || "Completed"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.overdue") || "Overdue"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.avgTimeDays") || "Avg Time (Days)"}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.taskAnalysis.completionRateLabel") || "Completion Rate"}
                </th>
              </tr>
            </thead>
            <tbody className='bg-white divide-y divide-gray-200'>
              {sortedTechnicians.map((tech) => (
                <tr key={tech.technicianId}>
                  <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900'>
                    <div>
                      {tech.technicianName}
                      {tech.isExternal && (
                        <span className='ml-2 text-xs text-blue-600'>
                          [{t("analysis.taskAnalysis.external") || "External"}]
                        </span>
                      )}
                    </div>
                    <div className='text-xs text-gray-500'>{tech.technicianEmail}</div>
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {tech.isExternal ? (
                      <span className='text-blue-600'>
                        {tech.companyName || t("analysis.taskAnalysis.external") || "External"}
                      </span>
                    ) : (
                      <span className='text-gray-600'>{t("analysis.taskAnalysis.internal") || "Internal"}</span>
                    )}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {tech.total}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {tech.completed}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-red-600'>
                    {tech.overdue}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {tech.avgCompletionTime > 0
                      ? tech.avgCompletionTime.toFixed(1)
                      : "—"}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm'>
                    <div className='flex items-center gap-2'>
                      <div className='flex-1 bg-gray-200 rounded-full h-2 max-w-[100px]'>
                        <div
                          className={`h-2 rounded-full ${
                            tech.completionRate >= 80
                              ? "bg-green-500"
                              : tech.completionRate >= 50
                              ? "bg-orange-500"
                              : "bg-red-500"
                          }`}
                          style={{ width: `${Math.min(tech.completionRate, 100)}%` }}
                        />
                      </div>
                      <span
                        className={`font-medium ${
                          tech.completionRate >= 80
                            ? "text-green-600"
                            : tech.completionRate >= 50
                            ? "text-orange-600"
                            : "text-red-600"
                        }`}
                      >
                        {tech.completionRate.toFixed(1)}%
                      </span>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </>
  );
}

